package easik;

import java.util.LinkedList;

import easik.states.BasicEditingState;
import easik.states.EasikState;



/**
 *	The state manager is a class that keeps track of the current state of the 
 *	program. Different states are used in order to interpret more complicated
 *	inputs. An example of this would be the process needed to create a 
 *	commutative diagram. First there are several clicks required to select each
 *	path. Paths can listen to what's going on with the UI, and can change and 
 *	collect information that way.
 * 
 * The state manager works like a stack. New states are pushed on top and then
 * the state manager forwards any important messages up to the active state.
 *  
 * @author Rob Fletcher 2005
 * @author Vera Ranieri 2006
 * @version 2006-05-30 Vera Ranieri
 */
public class StateManager {
	/**
	 * Boolean to record whether the user has pressed the <it>Finish</it> button.
	 */
	private boolean _finished;
	/**
	 * The stack.
	 */
	private LinkedList<EasikState> _states = new LinkedList<EasikState>();
	
	/**
	 * Initialize the state manager
	 * @param inApp The current instance of this application
	 */
	public StateManager(Easik inApp) {
		pushState( new BasicEditingState());
		_finished = false;
		updateStatus();
	}
	
	/**
	 * Hook for informing the current active state that the selection has 
	 * been updated
	 */
	public void selectionUpdated() {		
		((EasikState)_states.getFirst()).selectionUpdated();		
	}
	
	/**
	 * Non destructively take a look at the top item on the stack.
	 * @return The top state
	 */
	public EasikState peekState() {
		return (EasikState)_states.getFirst();
	}
	
	/**
	 * Pop the state off of the top of the stack. After the state has been popped
	 * the 'popped' hook is called. 
	 * @return The popped state
	 */
	public EasikState popState() {
		EasikState toReturn = (EasikState)_states.removeFirst() ;		
		// Inform the state it has been popped
		toReturn.poppedOff();
		// Inform the new top state that it is in the spotlight
		peekState().gotFocus();		
		// Update the labels based on the new state
		updateStatus();		
		return toReturn;
	}
	
	/**
	 * Pushes a state onto the stack, calls the 'pushed on' hook after being
	 * added.
	 * @param toPush The new state being pushed on
	 */
	
	public void pushState( EasikState toPush ) {
		_states.addFirst( toPush );
		toPush.gotFocus();		
		toPush.pushedOn();
		updateStatus();		
	}
	
	/**
	 * When called, this sends an updated state status string to the
	 * status bar
	 *
	 */
	public void updateStatus() {
		String newStatus = "";		
		for(int i = 0; i  < _states.size(); i++) {
			newStatus = " " + _states.get(i) + " >" + newStatus;
		}
		Easik.getInstance().getFrame().setStatusBar( newStatus );
	}
	
	/**
	 * Informs the top element that the next button has been clicked 
	 *
	 */
	public void nextClicked() {
		peekState().nextClicked();		
	}
	
	/**
	 * Informs the top element that the cancel button has been clicked. 
	 *
	 */
	public void cancelClicked() {
		peekState().cancelClicked();		
	}
	
	/**
	 * Informs the top element that the finish button has been clicked.
	 *
	 *@since 2006-05-30 Vera Ranieri
	 */
	public void finishClicked(){
		_finished = true;
		peekState().finishClicked();
	}
		
	/**
	 * Gets the value currently stored in <it>finsihed</it>
	 * @return True if the user has selected finished, false otherwise.
	 * @since 2006-05-30 Vera Ranieri
	 */
	public boolean getFinished(){
		return _finished;
	}
	
	/**
	 * Resets the value of <it>finsihed</it> to false
	 * @since 2006-05-30 Vera Ranieri
	 */
	public void resetFinished(){
		_finished = false;
	}
}
